/************************************************************************
 * @file: CThread.h
 *
 * @version: 0.1
 *
 * @description: This header file contains definition for Base class CThread
 * to create and register a thread. CThread class will be used by BackendAlsa
 * class for handling BackendAlsa class worker Thread.
 *
 * @component: platform/abalsa
 *
 * @author: Jens Lorenz (jlorenz@de.adit-jv.com) 2015
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#ifndef _ADIT_UTILITY_CTHREAD_H_
#define _ADIT_UTILITY_CTHREAD_H_

#include <string>
#include <pthread.h>
#include <sched.h>

#define PTHREAD_NAME_LEN    15

namespace adit
{

namespace utility
{

class CThread
{
public:
    struct __cpu_sched
    {
        int policy;
        int priority;

        __cpu_sched() : policy(SCHED_OTHER), priority(0) {};
        __cpu_sched(int pol, int prio) : policy(pol), priority(prio) {};
    };

private:
    typedef enum __state
    {
        STATE_JOINED   = 0x01,
        STATE_FORKING  = 0x02,
        STATE_RUNNING  = 0x04,
        STATE_STOPPING = 0x08,
        STATE_STOPPED  = 0x10,
        STATE_JOINING  = 0x20
    } eState;

    pthread_t               mId;
    std::string             mName;
    __cpu_sched             mCpuSched;
    eState                  mState;
    int                     mThreadErr;

    pthread_mutex_t         mMtx;
    pthread_cond_t          mCond;

public:
    CThread();
    virtual ~CThread();

    /**
     * @brief This function sets the thread name.
     * @param[in] name name of thread
     */
    void setThreadName(const std::string & name);

    /**
     * @brief This function sets the scheduling mode.
     * @param[in] name name of thread
     */
    void setThreadSched(const __cpu_sched & cpuSched);
    void setThreadSched(const int policy, const int priority);

    /**
     * @brief This function has to be called to start the thread initially.
     * Following sequence will be executed afterwards:
     *   if (0 == @see initThread())
     *      while (@see workerThread() == 0)
     *   @see deinitThread()
     *
     * @return 0 on success or <0 on failure
     */
    int startThread();

    /**
     * @brief This function stops the working thread. Another startThread() will continue the working.
     * @return 0 on success or <0 on failure
     */
    int stopThread();

    /**
     * @brief This function stops and joins the execution thread.
     * @return result of initThread() or workerThread()
     */
    int joinThread();

protected:
    /**
     * @brief This function will be called on startThread() of thread and can be used to allocated worker specific resources.
     * In case the resource allocation fails the function shall return != 0 which will directly call deinitThread().
     * @return 0 on success, other on error which will lead to end of thread execution
     */
    virtual int initThread() = 0;

     /**
      * @brief This function will be called after initThread() was successfully started.
      * This is the worker function and will be called periodically until:
      * a) joinThread() was called or
      * b) the function returns != 0.
      * @return 0 on success, other on error which will lead to end of thread execution
      */
    virtual int workerThread() = 0;

    /**
     * @brief This function will be called in case joinThread() was called or in case of failure in initThread() or workerThread().
     * It can be used to cleanup resources which were created within initThread().
     * @param[in] errInit return value of initThread() or workerThread()
     */
    virtual void deinitThread(int errInit) = 0;

private:
    int startWorkerThread();
    static void* _WorkerThread(void* arg) {
        static_cast<CThread*>(arg)->WorkerThread();
        return nullptr;
    };
    void WorkerThread();

    void setState(const eState state);
    eState getState();
    bool isStateForked();
    void waitForStateChange(const eState state);
    void setStateAndWaitForStateChange(const eState state);
};

} /* namespace utility*/

} /* namespace adit */

#endif /* _ADIT_UTILITY_CTHREAD_H_ */
